
C               FILE "EIGS" (SOURCE CODE: FORTRAN-90)
C    >>>> This version receives REAL(4), allocates REAL(8)  <<<<
C    >>>> Also, it allows transform of input A into Ginv[A] <<<<
C
C               Last revised:  25 April 2002
C
C *** This file is a package of subroutines that solves for eigenstructure.
C     The master subroutine, EIGS, is the only one called from outside the
C     package.  Only single-precision I/O arrays are received from the calling
C     program; all double-precision workspaces are internal allocations.  When
C     JOB calls return of the smallest eigenvalues, these and the corresponding
C     eigenvectors are returned in INCREASING order (smallest first).
C
C  Purpose:  Find the NF leading or trailing eigenvalues, and if wanted
C            the corresponding eigenvectors, of a symmetric real matrix A.
C            Also, option to compute the generalized inverse of A with or
C            without return of its eigenvectors.
C
C  Syntax:   CALL EIGS(NV, MF, A, LDA, RT, T, LDT, JOB, IER, MFILE)
C
C  Arguments:
C     NV     - Order of the matrix to be solved for eigenstructure. (Input)
C     MF     - Unless 0, the number NF of largest ( or smallest if JOB < 0 )
C              eigenvalues/eigenvectors wanted.  MF  0 flags NF = NV with
C              no return of eigvecs, so T in call can be a dummy. (Input)
C     A      - The order-NV symmetric real matrix to be analyzed. (Input/output)
C              Only the upper triangle of A is read, which can be sym-storage.
C              Not destroyed except when returning Ginv[A] in output.
C     LDA    - Leading dimension specified for A in the calling program. (Input)
C              If LDA = NV*(NV+1)/2, A is assumed to be in sym-storage.
C     RT     - Real vector returning the NF largest (smallest) eigenvalues
C              of matrix A in descending (ascending)  order. (Output)
C     T      - Real NV-by-NF matrix whose Kth column (K = 1,...,NF) is the
C              unit-norm eigenvector whose corresponding eigenvalue is RT(K).
C              (Output).  If any column of T is zero, solution for at least
C              one eigenvector failed to converge.
C     LDT    - Leading dimension specified for array T in the calling program's
C              dimension statement. (Input) If LDT=0, T is not returned.
C     JOB    - Job specification. (Input)
C                JOB  1 :  Return NF largest eigenvalues/eigenvectors
C                JOB = 0 :  Return NF largest eigenvalues; no eigenvectors
C                JOB = -1:  Return NF smallest eigenvalues; no eigenvectors
C                JOB = -2:  Return NF smallest eigenvalues/eigenvectors
C                JOB  NV:  Return Ginv[A] in A; also NV descending eigvals
C                             but omit eigvecs if either MF0 or LDT0.
C     IER    - 0, no problem; 1, EIVEC failure; 2, EIVAL failure; 3, both 1&2;
C              999, inadmissable input; -M, Ginv[A] has rank NY-M. (Output)
C     MFILE  - Unit number of the formatted file to which messages are to be
C              written. (Output)

      SUBROUTINE EIGS(NV,MF,AA,LDA,RTT,TT,LDT,JOB,IER,MFILE)
C Input shape: Input AA can be either full square or sym-storage.
C              Default is square; LDA = LO(NV,NV) flags sym-storage.
C       ****** WARNING: Flag can create error if true LDA exceeds NV by half
C  ALLOCATION should be pointless except for TT
      CHARACTER CF*12
      REAL(4) :: AA(*), TT(LDT,*), RTT(*)
      REAL(8) :: A(NV,NV), T(NV,NV), RT(NV), DTOL
      COMMON /CF/ CF
      LO(I,J) = J*(J-1)/2 + I    ! For sym-storage input
      LSQ(I,J) = LDA*(J-1) + I   ! For full-matrix input
      NF = MF; IF(MF<=0) NF = NV
      IF(NV<1) THEN
        WRITE(6,'(/" Matrix order NV =",I3," sent to EIGS ",
     +    "is inadmissibly small.")') NV; IER = 999
      ELSE IF(NF>NV) THEN
        WRITE(6,'(/" EIGS has been asked for NF =",I3," eigenvalues,",
     +    " exceeding the matrix order NV =",I4)') NF, NV; STOP
      ELSE IF(JOB<-2 .OR. (JOB>1.AND.JOB<NV)) THEN
        WRITE(6,'(/" Job call",I3," to EIGS is inadmissible. JOB mus",
     +    "t be in range [-2,1] or ",I3)') JOB, NV; IER = 999
      END IF
      JBB = JOB; IF(JOB>=0) JBB = JBB+1 ! Critical features of JBB are size/sign
      LL = 1; IF(LDA==LO(NV,NV)) LL = 0 ! LL=1 flags square storage
C Copy upper triangle of AA to up-triangle of square double-precision A
      A = 0.
      DO J = 1,NV
        DO I = 1,J
          IF(LL==0) A(I,J) = AA(LO(I,J))    ! Sym-storage input
          IF(LL==1) A(I,J) = AA(LSQ(I,J))   ! Square input
          A(J,I) = A(I,J)  ! Need full symmetric matrix
        END DO
      END DO
      CALL EIGSOL(NV,NF,A,RT,T,JBB,IER,MFILE)
      DO J = 1,NF
        RTT(J) = SNGL(RT(J))
      END DO
      IF(ABS(JBB)==1) RETURN
      IF(MF<=0 .OR. LDT<=0) GOTO 11
      DO J = 1,NF
        DO I = 1,NV
          TT(I,J) = SNGL(T(I,J))
        END DO
      END DO
11    IF(NF<NV .OR. JBB<NV) RETURN
      DTOL = 1.D-12   ! <****> Smallest relative DP discrimination is .22D-15
      NR = NV+1; A = 0.D0; NZ = 0
21    NR = NR-1; IF(RT(NR)<-DTOL) NZ = NZ+1; IF(RT(NR)<DTOL) GOTO 21
      IF(NZ>0) THEN
        WRITE(MFILE,'(/" NOTE: Some results here derive from an appr",
     +  "oximate generalized inverse wherein"/7X,"the",I2," negative",
     +  " eigenvalues (smallest,",E9.3,") of an")') NZ, RT(NV)
        IF(NV<10) WRITE(MFILE,'(7X,"order",I2," matrix are treated",
     +    " as zero.")') NV
        IF(NV>9.AND.NV<100) WRITE(MFILE,'(7X,"order",I3," matrix are",
     +    " treated as zero.")') NV
        IF(NV>99) WRITE(MFILE,'(7X,"order",I3," matrix are treated",
     +    " as zero.")') NV
      END IF
C Compute the Ginv of A.  NOTE: Axes with negative eivals are also discarded
      DO J = 1,NV
        DO I = 1,J
          DO K = 1,NR
            A(I,J) = A(I,J) + (T(I,K)*T(J,K))/RT(K)
          END DO
        END DO
      END DO
      DO J = 1,NV
        DO I = 1,J
          IF(LL==0) AA(LO(I,J)) = A(I,J)
          IF(LL==1) AA(LSQ(I,J)) = A(I,J)
          IF(LL==1) AA(LSQ(J,I)) = A(I,J)
        END DO
      END DO
      END SUBROUTINE
C
      SUBROUTINE EIGSOL(NV,NF,A,RT,T,JBB,IER,MFILE)
C  Purpose:  Execute the eigenstructure solution by calling appropriate
C            subroutines and, if necessary, evading nonconvergence of
C            eigenvectors by incrementing the diagonal of the analyzed matrix.
      REAL(8) A(NV,*), T(NV,*), RT(*), DDD(NV), EEE(NV), WE3(NV),
     +                 WK1(NV), WK2(NV), WK3(NV), WK4(NV), WK5(NV)
      REAL(8) EPS, ADD, AD0, DA, Z, S, X
      INTEGER IND(NV), JBB
      EPS = EPSILON(1.D0); IER = 0
      ADD = 0.0
C Rotate A to tri-diagonality
      CALL ROT3D(NV,A,NV,DDD,EEE,WE3)  ! (N,A,NV,D,E,E2)
      DO I = 1,NV
        WK1(I) = DDD(I); WK2(I) = EEE(I)
      END DO
C Compute eigenvalues
      CALL EIVAL(NV,NF,EPS,DDD,EEE,WE3,WK3,IND,JBB,IER)  ! JBB<0 calls smallest vals
C       Order test: JBB=+/- returns largest/smallest eivs descending/ascending
      DO I = 1,NF   ! Transfer eigenvalues to RT
        RT(I) = DDD(I)
      END DO
      JER = IER
      IF(ABS(JBB)==1) THEN  ! Otherwise, bad JER reported after EIVEC call
        IF(JER/=0) WRITE(MFILE,'(/" On EIGS call, solution did not",
     +    " converge for all eigenvalues.")'); RETURN
      END IF
15    IF(JBB>0) THEN   ! Call for largest eigenvalues
        DO I = 1,NF            ! **** Doesn't affect computation; Why needed ???
          RT(I) = DDD(NF+1-I)  !      JBB>0 reverses order again below
        END DO                 ! %% can omit if %% below is omitted
        NH  = NF/2
        KB = NF + 1
        DO K = 1,NH
          KB = KB - 1
          IX = IND(K)
          IND(K) = IND(KB)
          IND(KB) = IX
        END DO
      END IF
      DO I = 1,NV
        DDD(I) = WK1(I); EEE(I) = WK2(I)
      END DO
100   CALL EIVEC(NV,NF,RT,T,NV,DDD,EEE,WE3,WK1,WK2,WK3,WK4,WK5,IND,
     +           EPS,IER)
      IF(IER==0 .OR. ADD>60) GOTO 20
      AD0 = ADD; ADD = (ADD+1)*2; DA = ADD - AD0
      WRITE(6,'(4X,"Eigensolution not converging; will try ridge ",
     +  "increment",F4.0)') ADD
      DO I = 1,NV
        RT(I) = RT(I) + DA
        DDD(I) = DDD(I) + DA
      END DO
      GOTO 100
20    CALL UNROT(NV,NF,A,NV,T,EEE)
      IF(ADD<1.) GOTO 25
25    IF(JER/=0) WRITE(MFILE,'(/" On EIGS call, solution did not con",
     +  "verge for all eigenvalues.")')
      IF(IER>0) WRITE(MFILE,'(/" On EIGS call, five ridge increments f",
     +  "ailed to evade"/" nonconvergence of",I3," eigenvectors."/)')IER
      IF(ADD>0 .AND. IER==0) WRITE(MFILE,'(/" On EIGS call, initial ",
     +  "nonconvergence of eigenvectors was evaded by"/" ridge incre",
     +  "ment",F4.0/)') ADD
      IER = IER+JER
      J = 0
50    J = J + 1  ! Scale and orient eigenvectors
      X = 0.; Z = 0.; S = 0.
      DO K = 1,NV
        S = S + T(K,J); X = X + T(K,J)*T(K,J)
      END DO
      IF(X>1.0E-30) Z = 1./SQRT(X)
      IF(S<-1.0E-6) Z = -Z
      DO K = 1,NV
        T(K,J) = T(K,J)*Z
      END DO
      IF(J<NF) GOTO 50
      IF(JBB<0) RETURN
      NH = NF/2
      KB = NF + 1
      DO K = 1,NH  ! Reverse order of eigvals/eigvecs
        KB = KB - 1
        X = RT(K); RT(K) = RT(KB); RT(KB) = X   ! %% can omit if %% above is omitted
        DO I = 1,NV
          X = T(I,K); T(I,K) = T(I,KB); T(I,KB) = X
        END DO
      END DO
      END SUBROUTINE
C
      SUBROUTINE ROT3D(N,A,NV,D,E,E2)
C  Purpose:    Reduce a real symmetric matrix to tridiagonality
C              by orthogonal rotation.
      REAL(8)  A(NV,*), D(*), E(*), E2(*)
      REAL(8)  SCALE, F, G, H, X
      DO I = 1,N
        D(I) = A(N,I); A(N,I) = A(I,I)
      END DO
      BIG: DO I = N,1,-1
        L = I-1
        H = 0.
        IF(L>0) GOTO 5  ! L=0 when I=1
           E(I) = 0.0
           E2(I) = 0.0
           CYCLE BIG
5       SCALE = 0.0
        DO J = 1,L
          SCALE = SCALE + ABS(D(J))
        END DO
        IF(SCALE > 1.17577E-38) GOTO 25
        DO J = 1,L
          D(J) = A(L,J); A(L,J) = A(I,J)
        END DO
        DO J = 1,L
          A(I,J) = 0.0
        END DO
        E(I) = 0.0
        E2(I) = 0.0
        CYCLE BIG
25      DO J = 1,L
          D(J) = D(J)/SCALE
        END DO
        H = 0.0
        DO J = 1,L
          H = H + D(J)**2
        END DO
        E2(I) = SCALE*SCALE*H
        F = D(L)
        G = -SIGN(SQRT(H),F)
        E(I) = SCALE*G
        H = H - F*G
        D(L) = F - G
        IF(L==1) GOTO 75
      DO J = 1,L !  E = A'*D !  Doesn't override E(I) above because L < I
        E(J) = 0.
      END DO
      DO J = 1,L
        DO K = 1,J-1
          E(K) = E(K) + D(J)*A(J,K)
        END DO
        DO K = J,L
          E(K) = E(K) + D(J)*A(K,J)
        END DO
      END DO
        DO J = 1,L
          E(J) = E(J)/H
        END DO
        X = 0.0
        DO J = 1,L
          X = X + E(J)*D(J)
        END DO
        H = X/(H+H)
        DO J = 1,L
          E(J) = E(J) - H*D(J)
        END DO
        DO J = 1,L
          DO K = 1,J
            A(J,K) = A(J,K) - D(J)*E(K) - D(K)*E(J)
          END DO
        END DO
75      DO J = 1,L
          F = D(J); D(J) = A(L,J)
          A(L,J) = A(I,J)
          A(I,J) = F*SCALE
        END DO
      END DO BIG
      RETURN
      END SUBROUTINE
C
      SUBROUTINE EIVAL(N,NVAL,EPS,D,E,E2,E3,IND,JBB,IER)
C  Purpose:    Compute the largest NVAL eigenvalues, or smallest if
C              JBB is negative, of a symmetric tridiagonal matrix.
C              E2,E3 are locally initialized workspaces
      INTEGER  IND(*)
      REAL(8)  D(*), E(*), E2(*), E3(*)
      REAL(8) DELTA, EPS, EPS1, EP, ERR, F, P, Q, QP, R, S, TOT
      EPS1 = 0.0; IER = 0
      DO I = 1,N
        IF(ABS(E(I))>=1.0E-19) THEN
          E3(I) = E(I)**2
        ELSE
          E3(I) = 0.
        END IF
      END DO
      IF(JBB>0) THEN  ! Negate D-elements when largest are values wanted
        DO I = 1,N
          D(I) = -D(I)
        END DO
      END IF
      ERR = 0.0
      S = 0.0
      TOT = D(1)
      Q = 0.0
      J = 0
      DO I = 1,N  ! Loop IND
        P = Q
        IF(I==1) THEN
          E3(I) = 0.0
        ELSE
          IF(P<=EPS*( ABS(D(I))+ABS(D(I-1)) )) E3(I) = 0.0
        END IF
        IF(E3(I)==0.0) J = J + 1
        IND(I) = J
        Q = 0.0
        IF(I/=N) Q = ABS(E(I+1))
        TOT = MIN(D(I)-P-Q,TOT)
      END DO
      DO I = 1,N
        E2(I) = E3(I)
      END DO
      DO I = 1,N
        D(I) = D(I) - TOT
      END DO
      BIG: DO K = 1,NVAL
30      TOT = TOT + S
        DELTA = D(N) - S
        I = N
        F = EPS*ABS(TOT)
        IF(EPS1<F) EPS1 = F
        IF(DELTA<=EPS1) GOTO 70
        IF(K==N) GOTO 41
        DO J = K+1,N
          IF(E3(J) <= (EPS*ABS(D(J)+D(J-1)))**2) E3(J) = 0.0
        END DO
41      F = E3(N)/DELTA
        QP = DELTA + F
        P = 1.0E0
        DO  I = N-1,K,-1
          Q = D(I) - S - F
          R = Q/QP
          P = P*R + 1.0E0
          EP = F*R
          D(I+1) = QP + EP
          DELTA = Q - EP
          IF(DELTA<=EPS1) GOTO 70
          F = E3(I)/Q
          QP = DELTA + F
          E3(I+1) = QP*EP
        END DO
        D(K) = QP
        S = QP/P
        IF(TOT+S>TOT) GOTO 30
        IER = 2
        S = 0.0
        DELTA = QP
        DO J = K,N
          IF(D(J)>DELTA) CYCLE
          I = J
          DELTA = D(J)
          CONTINUE
        END DO
70      IF(I<N) E3(I+1) = E3(I)*F/QP
        II = IND(I)
        DO J = I-1,K,-1
          D(J+1) = D(J) - S
        END DO
        DO J = I,K+1,-1
          E3(J) = E3(J-1)
        END DO
        IF(I==K) GOTO 85
        DO J = I-1,K,-1
          IND(J+1) = IND(J)
        END DO
85      D(K) = TOT
        ERR = ERR + ABS(DELTA)
        E3(K) = ERR
        IND(K) = II
      END DO BIG
      IF(JBB<0) RETURN  ! Don't refect D when smallest values are wanted
      DO I = 1,N        ! Undo negation done earlier; ascending smallest
        D(I) = -D(I)    !   during solution now hecome descending largest
      END DO
      END SUBROUTINE
C
      SUBROUTINE EIVEC(N,NVEC,EVAL,EVEC,LVEC,D,E,E2,WK1,WK2,WK3,
     +                 WK4,WK5,IND,EPS,IER)
C  Purpose:    Compute the eigenvectors of a symmetric tridiagonal
C              matrix corresponding to an ordered set of eigenvalues.
      INTEGER    IND(*)
      REAL(8) EVAL(*), EVEC(LVEC,*), D(*), E(*), E2(*), WK1(*),
     +        WK2(*), WK3(*), WK4(*), WK5(*)
      REAL(8) EPS, EPS2, EPS3, EPS4, ANORM, VNORM, U, UK, V, X0, X1, XU
      E2(1) = 0.0; IER = 0
      ITAG = 0
      IQ = 0
      ANORM = ABS(D(1))
      DO I = 1,N
        ANORM = MAX(ANORM,ABS(D(I))+ABS(E(I)))
      END DO
30    KP = IQ + 1
      DO IQ = KP,N            !  Picks <KP,IQ> to enter loop BIG
        IF(IQ==N) GOTO 50
        IF(E2(IQ+1)==0.0) GOTO 50
      END DO
50    CONTINUE
      ITAG = ITAG + 1
      IS = 0
      BIG: DO IR = 1,NVEC
        IF(IND(IR)/=ITAG) CYCLE BIG
        ITS = 1
        X1 = EVAL(IR)
        IF(IS==0) THEN
          XU = 1.0E0
          IF(IQ-KP+1==1) THEN
             WK5(KP) = 1.0E0
             GO TO 130
          END IF
          VNORM = ABS(D(KP))
          DO I = KP+1,IQ
             VNORM = MAX(VNORM,ABS(D(I))+ABS(E(I)))
          END DO
          EPS2 = 1.0E-3*VNORM
          EPS3 = EPS*VNORM
          UK = IQ - KP + 1
          EPS4 = UK*EPS3
          UK = EPS4/SQRT(UK)
          IS = KP
          IGROUP = 0
        ELSE
          IF(ABS(X1-X0)<EPS2) THEN
            IGROUP = IGROUP + 1
            IF(X1<=X0) X1 = X0 + EPS3
          ELSE
            IGROUP = 0
          END IF
        END IF
        V = 0.0
        DO I = KP,IQ
          WK5(I) = UK
          IF(I==KP) THEN
            U = D(I) - X1 - XU*V
            IF(I/=IQ) V = E(I+1)
          ELSE
            IF(ABS(E(I))<ABS(U)) THEN
               XU = E(I)/U
               WK4(I) = XU
               WK1(I-1) = U
               WK2(I-1) = V
               WK3(I-1) = 0.0
               U = D(I) - X1 - XU*V
               IF(I/=IQ) V = E(I+1)
            ELSE
               XU = U/E(I)
               WK4(I) = XU
               WK1(I-1) = E(I)
               WK2(I-1) = D(I) - X1
               WK3(I-1) = 0.0
               IF(I/=IQ) WK3(I-1) = E(I+1)
               U = V - XU*WK2(I-1)
               V = -XU*WK3(I-1)
            END IF
          END IF
        END DO
        IF(ABS(U)<EPS*EPS3) U = EPS3
        WK1(IQ) = U
        WK2(IQ) = 0.0
        WK3(IQ) = 0.0*WK3(IQ)
80      DO I = IQ,KP,-1
          WK5(I) = (WK5(I)-U*WK2(I)-V*WK3(I))/WK1(I)
          V = U
          U = WK5(I)
        END DO
        IF(IGROUP==0) GOTO 101
        A: DO J = IR-1,IR-IGROUP
          IF(IND(J)/=ITAG) CYCLE A
          XU = 0.0
          DO I = KP,IQ
            XU = XU + WK5(I)*EVEC(I,J)
          END DO
          DO I = KP,IQ
            WK5(I) = WK5(I) - XU*EVEC(I,J)
          END DO
        END DO A
101     VNORM = 0.0
        DO I = KP,IQ
          VNORM = VNORM + ABS(WK5(I))
        END DO
        IF(VNORM>=1.0) GOTO 130
        IF(ITS<5) THEN
          IF(VNORM<=EPS3) THEN
             WK5(IS) = EPS4
             IS = IS + 1
             IF(IS>IQ) IS = KP
          ELSE
            XU = EPS4/VNORM
            DO I = KP,IQ
              WK5(I) = XU*WK5(I)
            END DO
          END IF
          DO I = KP+1,IQ
            U = WK5(I)
            IF(WK1(I-1)==E(I)) THEN
              U = WK5(I-1)
              WK5(I-1) = WK5(I)
            END IF
            WK5(I) = U - WK4(I)*WK5(I-1)
          END DO
          ITS = ITS + 1
          GO TO 80
        END IF
        IF(IER==0) XU = 0.0
        IER =  1
130     DO I = 1,N
          EVEC(I,IR) = 0.0
        END DO
        DO I = KP,IQ
          EVEC(I,IR) = EVEC(I,IR) + XU*WK5(I)
        END DO
        X0 = X1
      END DO BIG
      IF(IQ<N) GOTO 30
      END SUBROUTINE
C
      SUBROUTINE UNROT(N,NVEC,A,NV,EVEC,E)
C  Purpose:    Compute the eigenvectors of a real symmetric matrix
C              from the eigenvectors of its tridiagonal rotation.
      REAL(8)  A(NV,*), EVEC(NV,*), E(*), S
      IF(N==1) RETURN
      LP: DO I = 1,N
        IF(ABS(E(I))<1.0E-36) CYCLE LP
        DO J = 1,NVEC
          S = 0.0
          DO K = 1,I-1
            S = S + A(I,K)*EVEC(K,J)  ! Use lower triangle of A
          END DO
          S = (S/A(I,I-1))/E(I)       ! Use lower triangle of A
          DO K = 1,I-1
            EVEC(K,J) = EVEC(K,J) + S*A(I,K)
          END DO
        END DO
      END DO LP
      END SUBROUTINE

